/*
 * 	sample.t				version 1.0
 *	by Kevin Forchione		( Lysseus@msn.com )
 *	Copyright (c) 1999. 	All Rights Reserved.
 *
 *  SAMPLE DEMONSTRATION GAME FOR SENSE.T
 *
 *  This file demonstrates the containment class of sense.t
 *
 *  This file requires HTML TADS 2.5.1 or later.
 *
 *  You may modify and use this file in any way you want, provided that
 *	if you redistribute modified copies of this file in source form, the
 *  copies must include the original copyright notice (including this
 *  paragraph), and must be clearly marked as modified from the original
 *  version.
 *
 *------------------------------------------------------------------------------
 *		Revision History
 *------------------------------------------------------------------------------
 *		18-Jan-00:	Creation.
 */

/*
 *   Set a flag indicating that we want to use the HTML version of the
 *   status line code.  This must be set before we include adv.t, because
 *   adv.t needs to know what kind of status line we want. 
 */
#define USE_HTML_STATUS

/*
 *   Set another flag, this time indicating that we want to use the HTML
 *   version of the command prompt display code.  This code is in std.t,
 *   so we must define this flag before we include std.t.  
 */
#define USE_HTML_PROMPT

/*  For enterable objects we use the containment class! */
#define USE_CONTAINMENT

/*
 *   Include the TADS "adventure" library.  This library defines object
 *   classes that we can use as building blocks for creating the game
 *   objects.
 */
#include <adv.t>

/*
 *   Include the TADS "standard" library.  This contains default object
 *   definitions for several objects that are required in any TADS game.
 *   As you customize your game, you might eventually want to replace the
 *   definitions in std.t with your own definitions, but we can at least
 *   get started with the default versions.  
 */
#include <std.t>
#include <sense.t>
#include <doors.t>

#pragma C+

/*
 *   Replace the function "commontInit" defined in std.t so that we turn
 *   on HTML mode.
 */
replace commonInit: function
{
    /* display the special code sequence to turn on HTML recognition */
    "\H+";
}


/*
 *   We must define the initial location, where the player starts the
 *   game.  By the convention used in std.t, this location is called
 *   "startroom".
 *   
 *   We'll make startroom an object of class "room", which is the basic
 *   class defined in adv.t for a location in the game.  
 */
startroom: containment
    lightsOn = true
    /* 
     *   The "sdesc" is the short description of the location.  This is
     *   displayed when the player enters or re-enters the location. 
     */
    sdesc = "Entryway"

    /* 
     *   The "ldesc" is the long description of the location.  This is
     *   displayed when the player enters the location for the first time,
     *   or types "look" while in the location. 
     */
    ldesc =
    {
        "This large, formal entryway is slightly intimidating:
        the walls are lined with somber portraits of gray-haired
        men from decades past; a medieval suit of armor";

        /* if the axe is in the armor, list it specially */
        if (axe.isIn(suitOfArmor))
            ", posed with battle axe at the ready,";

        " towers over a single straight-backed wooden chair.
        The front door leads back outside to the south.  A
        hallway leads north.";
    }

    /* 
     *   To the north is the hallway.  Set the "north" property to the
     *   destination room object.  Other direction properties that we
     *   could set: east, west, north, up, down, plus the diagonals: ne,
     *   nw, se, sw.  We can also set "in" and "out".  
     */
    north = hallway

    /* 
     *   To the south is the front door; set the "south" property to refer
     *   to the door object.
     */
    south = frontDoor

    /* 
     *   set the "out" direction to be the same as south, since going
     *   south leads outside 
     */
    out = frontDoor
;

/*
 *   Define the front door.
 */
frontDoor: door
    foundin = [ startroom, frontYard ]
    noun = 'door'
    adjective = 'front'
    sdesc = "front door"
;

chessBoard: containment
    issurface = true
    location = startroom
    noun = 'board' 'chessboard'
    adjective = 'chess' 'checker' 'chequer'
    sdesc = "chess board"
;

redQueen: item
    location = chessBoard
    noun = 'queen'
    adjective = 'red'
    isHer = true
    sdesc = "red queen"
    ldesc = "She's a fierce little chess piece."
;

jar: containment
    iscontainer = true
    location = startroom
    noun = 'jar'
    adjective = 'lidless' 'ceramic'
    sdesc = "lidless ceramic jar"
;

candle: lightsource
    location = jar
    noun = 'candle'
    sdesc = "candle"
    islit = true
;

bag: containment
    isopenable = true
    location = startroom
    noun = 'bag'
    sdesc = "bag"
    insideDesc = "%You're% inside a close confining space."
    ldesc = {
        "It looks like a sleeping bag. ";
        pass ldesc;
    }
;

joe: Actor
    location = startroom
    noun = 'joe'
    sdesc = "Joe"
    adesc = "Joe"
    thedesc = "Joe"
    isHim = true
    actorDesc = "Joe is here, looking for lerts."
    actorAction(v, d, p, i) = {}
;

/*
 *   Define the chair.  The adv.t "chairitem" class is helpful here.
 */
chair: fixeditem, containment
    issurface = true
    isenterable = true
    location = startroom
    sdesc = "wooden chair"
    noun = 'chair'
    adjective = 'straight-backed' 'wooden'
;


/*
 *   Define the suit of armor.  It can't be moved, so it's a "fixeditem"
 *   (defined in adv.t).  
 */
suitOfArmor: fixeditem
    location = startroom
    sdesc = "suit of armor"
    noun = 'suit' 'armor'
    adjective = 'medieval'
    ldesc =
    {
        "It's a suit of plate-mail armor that looks suitable
        for a very tall knight. ";

        /* if I'm holding the axe, so note */
        if (axe.isIn(self))
            "The armor is posed with a huge battle-axe held
            at the ready. ";
    }

    /* 
     *   This is a little unusual, but set the armor to be a "quiet
     *   container."  We initially have the axe located in the armor, even
     *   though the armor isn't a container in the normal sense.  The
     *   "qcontainer" flag will keep the axe from being listed separately
     *   in the contents of the room. 
     */
    isqcontainer = true
;

/*
 *   Battle Axe 
 */
axe: item
    sdesc = "battle axe"

    /* it starts out in the suit of armor */
    location = suitOfArmor

    /* 
     *   Define vocabulary.  Note that we provide some synonyms for things
     *   that really could be separate objects (the blade, the dried blood
     *   on the blade), but which aren't important enough to be
     *   implemented separately. 
     */
    noun = 'axe' 'blood' 'blade' 'edge'
    adjective = 'steel' 'battle' 'dried'
    ldesc = "It's a large steel battle axe.  A little bit of
             dried blood on the edge of the blade makes the authenticity
             of the equipment quite credible. "
;

/*
 *   Define the portraits.  We don't want to define several individual
 *   portraits, so define a single object that refers to the portraits
 *   collectively. 
 */
portraits: fixeditem
    location = startroom
    noun = 'portraits' 'pictures' 'men'
    adjective = 'somber' 'gray-haired'
    sdesc = "portraits"

    /* 
     *   set the "isThem" flag, so that default messages in adv.t refer to
     *   this object in the plural
     */
    isThem = true

    /* 
     *   Define an "adesc": this is used in messages to refer to the
     *   object with an indefinite pronoun (such as "a" or "an").
     *   Normally, the adv.t base class would define this as "a
     *   <<sdesc>>", but this isn't appropriate for a collective object -
     *   "a portraits" doesn't sound right.
     */
    adesc = "a portrait"
;

////////////////////////////////////////////////////////////////////////
//
//  HALLWAY
//
////////////////////////////////////////////////////////////////////////

/*
 *   The hallway to the north of the starting room 
 */
hallway: containment
    lightsOn = true
    sdesc = "Hallway"
    ldesc = "This broad, dimly-lit corridor runs north and south. "

    south = startroom
    north = kitchen
;

hugeCupboard: containment
    islockable = true
    isenterable = true
    location = hallway
    noun = 'cupboard'
    adjective = 'huge'
    sdesc = "huge cupboard"
    insideDesc = "It's a snug little cupboard in here, 
        almost a room in itself."
;

plasticCabinet: containment
    isopenable = true
    istransparent = true
    isenterable = true
    location = hugeCupboard
    noun = 'cabinet'
    adjective = 'plastic'
    sdesc = "plastic cabinet"
    insideDesc = "The plastic walls of the cabinet distort the view."
;

pileOfClothes: item
    location = hugeCupboard
    noun = 'clothes'
    adjective = 'pile'
    sdesc = "pile of clothes"
;

////////////////////////////////////////////////////////////////////////
//
//  KITCHEN
//
////////////////////////////////////////////////////////////////////////

kitchen: containment
    lightsOn = true
    sdesc = "Kitchen"
    ldesc = "This is a surprisingly cramped kitchen, equipped with
             antiques: the stove is a huge black iron contraption,
             and there doesn't even seem to be a refrigerator.  A
             hallway lies to the south. "
    south = hallway
;

table: containment
    issurface = true
    isenterable = true
    location = kitchen
    noun = 'table'
    sdesc = "table"
    insideDesc = "The surface of the table is small and cramped."
;


bottle: containment
    istransparent = true
    location = table
    sdesc = "glass bottle"
    noun = 'bottle'
    adjective = 'glass'
;

apple: fooditem
    location = table
    noun = 'apple'
    sdesc = "apple"
    adesc = "an apple"
;

/*
 *   The stove is fixed, since we don't want the player to be able to move
 *   it, and we'd also like it to be a container that can be opened and
 *   closed, so make it both a fixeditem and an openable.
 */
stove: fixeditem, containment
    isopenable = true
    location = kitchen
    noun = 'stove' 'oven' 'contraption' 'door'
    adjective = 'huge' 'black' 'iron' 'stove'
    sdesc = "stove"
    ldesc =
    {
        "It's a huge black iron cube, with a front door that swings
        open sideways.  The door is currently <<
        self.isopen ? "open" : "closed">>. ";

        /* list my contents if there's anything inside */
        if (self.isopen && itemcnt(self.contents) != 0)
            "Inside the stove you can see <<listcont(self)>>. ";
    }

    /* it starts off closed */
    isopen = nil
;

/*
 *   A fooditem is something that we can eat 
 */
loaf: fooditem
    location = stove
    sdesc = "loaf of bread"
    ldesc = "It's a fresh loaf with a golden-brown crust. "
    noun = 'loaf' 'bread' 'crust'
    adjective = 'fresh' 'golden-brown' 'brown'

    /* 
     *   override the "doEat" method - this is called when the player eats
     *   the item 
     */
    doEat(actor) =
    {
        "You tear off a piece and eat it; it's delicious.  You tear off
        a little more, then a little more, and before long the whole loaf
        is gone. ";

        /* make the bread vanish by moving it to "nil" */
        self.moveInto(nil);
    }
;

////////////////////////////////////////////////////////////////////////
//
//  FRONT YARD
//
////////////////////////////////////////////////////////////////////////

frontYard: containment
    lightsOn = true
    toRiver = bendInRiver
    sdesc = "Front Yard"
    ldesc = "A broad swathe of new-mown grass, delineated by the
        sparkling-blue arc of a slow moving river as it wends past an  
        old Victorian-style house, sets the scene for this picturesque 
        view of the lawn. The front door of the house lies to the north."
    north = frontDoor
;

raft: containment
    isvehicle = true
    location = frontYard
    sdesc = "inflatable rubber raft"
    adesc = "an inflatable rubber raft"
    noun = 'raft'
    adjective = 'inflatable' 'rubber'
    isinflated = nil
    ldesc = "It's an inflatable rubber raft.  Currently,
        it's <<self.isinflated ? "inflated" : "not inflated">>. "
    verDoTake( actor ) =
    {
        if ( self.isinflated )
            "You'll have to deflate it first. ";
        else
            pass doTake;
    }
    verDoInflateWith( actor, iobj ) =
    {
        if ( self.isinflated) "It's already inflated! ";
    }
    doInflateWith( actor, iobj ) =
    {
        if ( self.isIn( actor ) )
            "You'll have to drop it first. ";
        else
        {
            "With some work, you manage to inflate the raft. ";
            self.isinflated = true;
        }
    }
    verDoDeflate( actor ) =
    {
        if ( !self.isinflated ) 
            "It's as deflated as it can get. ";
    }
    doDeflate( actor ) =
    {
        "You let the air out, and the raft collapses to a
        compact pile of rubber. ";
        self.isinflated = nil;
    }
    doBoard( actor ) =
    {
        if ( self.isinflated ) 
            pass doBoard;
        else 
            "You'll have to inflate it first. ";
    }
    doUnboard( actor ) =
    {
        if ( isclass( self.location, riverRoom ) )
            "Please keep your hands and arms inside the raft
            at all times while the raft is in motion. ";
        else
            pass doUnboard;
    }
    out =
    {
        if ( isclass( self.location, riverRoom ) )
        {
            "You can't get out until you've landed the raft. ";
            return( nil );
        }
        else
            pass out;
    }
    verDoLaunch( actor ) = {}
    doLaunch( actor ) =
    {
        if ( isclass( self.location, riverRoom ) )
            "You're already afloat, if you didn't notice. ";
        else if ( self.location.toRiver == nil )
            "There doesn't appear to be a suitable waterway here. ";
        else if ( parserGetMe().location != self )
            "You'll have to get in the raft first. ";
        else
        {
            "The raft drifts gently out into the river. ";
            notify( self, &moveDaemon, 0 );
            self.counter = 0;
            self.moveInto( self.location.toRiver );
        }
    }
    verDoLand( actor ) = {}
    doLand( actor ) =
    {
        if (not  isclass( self.location, riverRoom ) )
            "You're already fully landed. ";
        else if ( self.location.toLand == nil )
            "There's no suitable landing here. ";
        else
        {
            "You steer the raft up onto the shore. ";
            unnotify( self, &moveDaemon );
            self.moveInto( self.location.toLand );
        }
    }
    moveDaemon =
    {
        "\bThe raft continues to float downstream. ";
        self.counter++;
        if ( self.counter > 1 )
        {
            self.counter = 0;
            if ( self.location.downRiver == nil )
            {
                "The raft comes to the end of the river, and lands.";
                self.moveInto( self.location.toLand );
                unnotify( self, &moveDaemon );
            }
            else
            {
                self.moveInto( self.location.downRiver );
                self.location.riverDesc;
            }
        }
    }
;

inflateVerb: deepverb
    sdesc = "inflate" 
    verb = 'inflate' 'blow up'
    ioAction( withPrep ) = 'InflateWith'
    prepDefault = withPrep
;

deflateVerb: deepverb
    sdesc = "deflate"
    verb = 'deflate'
    doAction = 'Deflate'
;

pump: item
    sdesc = "pump"
    location = frontYard
    noun = 'pump'
    verIoInflateWith( actor ) = {}
    ioInflateWith( actor, dobj ) =
    {
        dobj.doInflateWith( actor, self );
    }
;

launchVerb: deepverb
    sdesc = "launch"
    verb = 'launch'
    doAction = 'Launch'
;

landVerb: deepverb
    sdesc = "land"
    verb = 'land'
    doAction = 'Land'
; 

////////////////////////////////////////////////////////////////////////
//
//  RIVER
//
////////////////////////////////////////////////////////////////////////

/*** FROM THE TADS AUTHORS MANUAL ***
Note that we'll expect each river room to have a property, riverDesc, 
which displays a message when the raft drifts into that room. 
The moveDaemon method will keep the raft in each river room for two turns, 
then move the raft to the next river room, calling riverDesc to note the entry. 
When the raft comes to the end of the river, the method will automatically land 
the raft; this means that the last river room must have a non-nil toLand property. 
You could alternatively put in a waterfall or other special effect when reaching 
the end of the river.

To build a river, all you have to do is define a series of rooms of class riverRoom, 
and point the downRiver property in each to the next room downriver. Landings are built 
by setting the toRiver and toLand properties of the landing room and corresponding river 
room, respectively, to point to each other.
*/

class riverRoom: containment
    lightsOn = true
    riverDesc = ""
    toRiver = nil
    toLand = nil
    downRiver = nil
    ldesc = { self.riverDesc; }
;

bendInRiver: riverRoom
    sdesc = "Bend In River"
    ldesc = "The river winds its way gently past your house."
    riverDesc = "The current is slow moving and serene at this spot along
        the river."
    toLand = frontYard
    downRiver = rapids
;

rapids: riverRoom
    sdesc = "Rapids"
    ldesc = "The waters rushing through the narrow canyons drown out all
        other sounds."
    riverDesc = "The water suddenly gets very choppy and dangerous
    here."
    downRiver = broadRiver
;

broadRiver: riverRoom
    sdesc = "Broad River"
    ldesc = "This could easily be a lake, the water is as calm as a mill-
        pond."
    riverDesc = "The river slows as it broadens out at this point."
    toLand = embankment
;

embankment: containment
    lightsOn = true
    sdesc = "Embankment"
    ldesc = "You're standing on the embankment of a broad river."
;
